TimeStart("Calculation")

-- For this tutorial, we will include a set of custom functions, that will make the code
-- shorter and increase readibility. The function dofile() simply appends the content of the 
-- specified file at the position where this function is called.
dofile("functions_Fabian.lua")


------------------------------------------------------------------------------------------------------------------------------------------
-- Initialize operators
------------------------------------------------------------------------------------------------------------------------------------------
-- For EuO, we will investigate the core level excitation from 3d to 4f.
-- This makes a total number of 24 spin orbitals. We define the 
NF=24
NB=0
IndexDn_3d = {0, 2, 4, 6, 8}
IndexUp_3d = {1, 3, 5, 7, 9}
IndexDn_4f = {10, 12, 14, 16, 18, 20, 22}
IndexUp_4f = {11, 13, 15, 17, 19, 21, 23}

-- The function get_operators_NF() creates all desired operators up to the Coulomb interaction
-- between two shells. One can enter an arbitrary number of shells. The last argument takes
-- a rotation matrix to directly receive the operators in the desired basis.
Opp = get_operators_NF(NF, {{"3d", IndexDn_3d, IndexUp_3d}, {"4f", IndexDn_4f, IndexUp_4f}}, nil)

-- The dipole operators have to be created manually
Akm = {{1,-1,sqrt(1/2)},{1, 1,-sqrt(1/2)}}
TXASx = NewOperator("CF", NF, IndexUp_4f, IndexDn_4f, IndexUp_3d, IndexDn_3d, Akm)

Akm = {{1,-1,sqrt(1/2)*I},{1, 1,sqrt(1/2)*I}}
TXASy = Chop(NewOperator("CF", NF, IndexUp_4f, IndexDn_4f, IndexUp_3d, IndexDn_3d, Akm))

Akm = {{1,0,1}}
TXASz = NewOperator("CF", NF, IndexUp_4f, IndexDn_4f, IndexUp_3d, IndexDn_3d, Akm)

-- to create the fundamental spectra, we also need an imaginary combination of the dipole operators
TXASxpiy =  sqrt(1/2)*(TXASx + I * TXASy)

-- to shorten the effective operators, we use the short hands:
Opp1 = Opp["1"]
Sx = Opp["Sx_4f"]
Sy = Opp["Sy_4f"]
Sz = Opp["Sz_4f"]
Ssqr = Opp["Ssqr_4f"]

-- Here, we define the effective operators. Luckily, Lua leaves the option to create tables
-- of any data type and define custom functions for their respective multiplications, etc.
OppR_A1g_Matrix = { {Opp1, 0, 0},
                    {0, Opp1, 0},
                    {0, 0, Opp1}}
                
OppR_T1u_Matrix = { {0, Sz, -Sy},
                    {-Sz, 0, Sx},
                    {Sy, -Sx, 0}}
                
OppR_Eg_Matrix = {  {2*(Sx*Sx - 1/3*Ssqr), 0, 0},
                    {0, 2*(Sy*Sy - 1/3*Ssqr), 0},
                    {0, 0, 2*(Sz*Sz - 1/3*Ssqr)}}
                
OppR_T2g_Matrix = { {0, (Sx*Sy + Sy*Sx), (Sz*Sx + Sx*Sz)},
                    {(Sx*Sy + Sy*Sx), 0, (Sy*Sz + Sz*Sy)},
                    {(Sz*Sx + Sx*Sz), (Sy*Sz + Sz*Sy), 0}}
------------------------------------------------------------------------------------------------------------------------------------------
------------------------------------------------------------------------------------------------------------------------------------------





------------------------------------------------------------------------------------------------------------------------------------------
-- Initialize Eu2+
------------------------------------------------------------------------------------------------------------------------------------------
-- We define variables for the interaction energies obtained from a Hartree-Fock calculation
Ryd = 13.6

F0_4f = 2.0398645*Ryd
F2_4f = 0.9503917*Ryd
F4_4f = 0.5926381*Ryd
F6_4f = 0.4253442*Ryd

F0_3d4f = 2.8421169*Ryd
F2_3d4f = 0.5868624*Ryd
F4_3d4f = 0.2665610*Ryd

G1_3d4f = 0.4024913*Ryd
G3_3d4f = 0.2354934*Ryd
G5_3d4f = 0.1625512*Ryd

-- 4f spin orbit coupling will be neglected in this tutorial. This is garantuees that
-- we keep our effective operators exact.
zeta_4f = 0 --0.01175*Ryd
zeta_3d = 0.82050*Ryd

Bz = 0.0000001

Hex = 0.1

-- EuO has a 4f7 configuration.
StartRestrictions = {NF, NB, {"1111111111 00000000000000",10, 10}, {"0000000000 11111111111111",7, 7}}

-- we define the 4f Hamiltonian without an exchange field and subtract the groundstate energy.
-- For the intermediate state Hamiltonian, we include the 3d interactions
Hamiltonian_0 = F0_4f*Opp["F0_4f"] + F2_4f*Opp["F2_4f"] + F4_4f*Opp["F4_4f"] + F6_4f*Opp["F6_4f"] + zeta_4f*Opp["ldots_4f"] + Bz*(Opp["Lz_4f"] + 2*Opp["Sz_4f"])
psi_0 = Eigensystem(Hamiltonian_0, StartRestrictions, 1)
Hamiltonian_0 = Hamiltonian_0 - psi_0*Hamiltonian_0*psi_0*Opp["1"]
XASHamiltonian_0 = F0_3d4f*Opp["F0_3d4f"] + F2_3d4f*Opp["F2_3d4f"] + F4_3d4f*Opp["F4_3d4f"] + G1_3d4f*Opp["G1_3d4f"] + G3_3d4f*Opp["G3_3d4f"] + G5_3d4f*Opp["G5_3d4f"] + zeta_3d*Opp["ldots_3d"]
------------------------------------------------------------------------------------------------------------------------------------------
------------------------------------------------------------------------------------------------------------------------------------------



------------------------------------------------------------------------------------------------------------------------------------------
-- Initialize HFM
------------------------------------------------------------------------------------------------------------------------------------------
-- In this script, we only want ot calculate the Heisenberg ferromagnet with
-- its one magnon and two magnon spectral function. Therefore, we need to define
-- lattice vectors and magnetic exchange constants that describe the hopping of
-- spin excitations along these directions.
lattices_EuO = {    
    {{ 1, 0, 0}},
    {{ 0, 1, 0}},
    {{ 0, 0, 1}},
    {{ 1/2, 1/2, 0}},
    {{ 0, 1/2, 1/2}},
    {{ 1/2, 0, 1/2}}, 
    {{ 1/2,-1/2, 0}},
    {{ 0, 1/2,-1/2}},
    {{-1/2, 0, 1/2}},
}

meV = 0.001
S = 7/2

-- we include the next nearest neighbor exchange along the simple cubic directions.
-- But as we are in a fcc lattice, the nearest neighbors sit at the 12 faces.
J_nn = 2*0.606*phys["kB"]*S/meV
J_nnn = 2*0.119*phys["kB"]*S/meV

-- To map the hoppings to the correct lattice vectors, we need to have the same order.
hoppings_EuO = {J_nnn, J_nnn, J_nnn, J_nn, J_nn, J_nn, J_nn, J_nn, J_nn}
----------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------



------------------------------------------------------------------------------------------------------------------------------------------
--Initialize calculation
------------------------------------------------------------------------------------------------------------------------------------------
NE1 = 1
Gamma1 = 1
-- We set the incident energy to the maximum of the M5 edge, we have obtained from the XAS spectrum.
Emin1 = 257.6 --240 --259.76
Emax1 = Emin1 + 0.1 --320
NPoints = 100
minPhi = 0
maxPhi = math.pi/2
NE = 100
Emin = 0
Emax = 12
gamma = 0.5
-- The Brillouin zone boundary is at 2pi
BZ = math.pi*2
Nq = 10
------------------------------------------------------------------------------------------------------------------------------------------
------------------------------------------------------------------------------------------------------------------------------------------






------------------------------------------------------------------------------------------------------------------------------------------
--Calculate fundamental spectra
------------------------------------------------------------------------------------------------------------------------------------------
-- Next, we create the fundamental spectra. As we are only interested in one incident energy, we can
-- save some computation time by setting NE1=1
Hamiltonian_z = Hamiltonian_0 + Hex*Opp["Sz_4f"]
XASHamiltonian_z = Hamiltonian_z + F0_3d4f*Opp["F0_3d4f"] + F2_3d4f*Opp["F2_3d4f"] + F4_3d4f*Opp["F4_3d4f"] + G1_3d4f*Opp["G1_3d4f"] + G3_3d4f*Opp["G3_3d4f"] + G5_3d4f*Opp["G5_3d4f"] + zeta_3d*Opp["ldots_3d"]

-- Also, we only need the ground state with the respective magnetic symmetry.
-- Thus, we only need to calculate one eigenstate.
-- Attention! Here we do not get a table but a single wave function
psi_z = Eigensystem(Hamiltonian_z, StartRestrictions, 1)

XAS_x = CreateSpectra(XASHamiltonian_z, TXASx, psi_z, {{"Emin",Emin1}, {"Emax",Emax1}, {"NE",NE1}, {"Gamma",Gamma1}})
XAS_y = CreateSpectra(XASHamiltonian_z, TXASy, psi_z, {{"Emin",Emin1}, {"Emax",Emax1}, {"NE",NE1}, {"Gamma",Gamma1}})
XAS_z = CreateSpectra(XASHamiltonian_z, TXASz, psi_z, {{"Emin",Emin1}, {"Emax",Emax1}, {"NE",NE1}, {"Gamma",Gamma1}})

XAS_xpiy = CreateSpectra(XASHamiltonian_z, TXASxpiy, psi_z, {{"Emin",Emin1}, {"Emax",Emax1}, {"NE",NE1}, {"Gamma",Gamma1}})

sigma = {}

-- we can extract the complex value for the first incident energy in the spectrum, if we transform it into table type.
sigma[0] = (Spectra.ToTable((XAS_x + XAS_y + XAS_z)/3))[1][1][2] --sigma0
sigma[1] = (Spectra.ToTable(I*(XAS_xpiy - 1/2*(XAS_x + XAS_y))))[1][1][2] --sigma1
sigma[2] = (Spectra.ToTable((XAS_z - 1/2*(XAS_x + XAS_y))))[1][1][2] --sigma2

-- Any Lua object can be saved as a text file during runtime. This is very useful to
-- for example save intermediate results. If errors occur later in the script, we can
-- load these textfile and skip the previous time consuming steps via dofile().
-- The argument needs to be identical to the object, you want to save.
io.put(directory, "sigma")
--dofile(directory."sigma.Quanty")
------------------------------------------------------------------------------------------------------------------------------------------
------------------------------------------------------------------------------------------------------------------------------------------






------------------------------------------------------------------------------------------------------------------------------------------
--Calculate lowest Eu2+ eigenstates
------------------------------------------------------------------------------------------------------------------------------------------
-- As we already have calculated the fundamental spectra, we now can calculate the RIXS Spectra for any
-- arbitrary spin direction and only need to calculate the lowest eigenstates of interest.
Hamiltonian = Hamiltonian_z
psiList = Eigensystem(Hamiltonian, StartRestrictions, 8)
print_expectation_values(psiList, {H = Hamiltonian, Sz = Opp["Sz_4f"], Lz = Opp["Lz_4f"], Jz = Opp["Jz_4f"], Ssqr = Opp["Ssqr_4f"], Lsqr = Opp["Lsqr_4f"], Jsqr = Opp["Jsqr_4f"]})
------------------------------------------------------------------------------------------------------------------------------------------
------------------------------------------------------------------------------------------------------------------------------------------


-- For this tutorial, we will create two output files.
-- one for the two dimensional output
-- and a second for the three dimensional spectral funcitons
OUT_k = io.open(directory .. "RIXS_spinwaves_EuO.dat", "w")
OUT_w_k = io.open(directory .. "RIXS_spinwaves_EuO_w_k.dat", "w")

-- we iterate over a set of points along our desired path
for i = 0, NPoints do
    
    -- the scattering geometry will be chose such that we only measure the magnon excitations
    -- along the kz direction. The only parameter will be the angle phi between the xy plane
    -- and the incident/ outgoing beam. For phi = 0 we scatter in plane. For phi = pi/2, we
    -- scatter along the [001] vector.
    phi = minPhi + i*(maxPhi - minPhi)/NPoints
    
    k_in =  {{ math.cos(phi), 0,-math.sin(phi)}}
    k_out = {{ math.cos(phi), 0, math.sin(phi)}} -- -k_out
    
    setmetatable(k_in, MatrixMeta)
    setmetatable(k_out, MatrixMeta)
    
    -- As we have chosen the incident energy to be at the M5 edge, we also need to include this
    -- for the calculation of our k vector. As the only difference |k_in| - |k_out| ~10meV,
    -- the energy loss can be ignored. As the XAS calculation does not reproduce real
    -- experiment energies, we need to shift for the correct k vector
    E_in = Emin1 + 872.4
    --E_in = 2*E_in
    hbar = 6.582119569e-16 --eVs Wiki
    c = 299792458 --m/s Wiki
    
    -- the lattice constant of EuO (here, nnn distance).
    a = 5.144e-10

    k_in = k_in*E_in/hbar/c*a
    k_out = k_out*E_in/hbar/c*a
    k = k_in - k_out
    
    -- to see the progress, we print the current k point and its absolute value with
    -- respect to the Brillouin zone boundary.
    print(string.format("%03i", i), "k_z in units 2*pi/a:", math.sqrt((k*Matrix.Transpose(k))[1][1])/BZ)
    
    -- this funciton calculates the single magnon dispersion.
    w_k = HFM_w_k(k, hoppings_EuO, lattices_EuO)
    
    OUT_k:write(string.format("%.15e %.15e ", i, w_k))
    
    -- for the polarizations, we choose linear pi (in scattering plane xz) for the incident beam
    -- and sigma (perpendicular) for the outgoing beam.
    e_in_pi =  {{ math.cos(phi + math.pi/2), 0,-math.sin(phi + math.pi/2)}}
    e_out_sigma =  {{ 0, 1, 0}}
    
    setmetatable(e_in_pi, MatrixMeta)
    setmetatable(e_out_sigma, MatrixMeta)
    
    -- the polarizations determine, which spin operators enter into the effective operators.
    -- here, we use a custom function to do data type independent matrix operations. The order 
    -- of multiplication is given by the order inside the argument. Furhter, the operators
    -- are multiplied by the aplitudes of the respective fundamental spectra.
    OppR_A1g = Matrix_multiply({Matrix.Conjugate(e_out_sigma), OppR_A1g_Matrix, Matrix.Transpose(e_in_pi)})*sigma[0]
    OppR_T1u = Matrix_multiply({Matrix.Conjugate(e_out_sigma), OppR_T1u_Matrix, Matrix.Transpose(e_in_pi)})*sigma[1]/S
    OppR_Eg  = Matrix_multiply({Matrix.Conjugate(e_out_sigma), OppR_Eg_Matrix, Matrix.Transpose(e_in_pi)})*sigma[2]/(S*(2*S - 1))
    OppR_T2g = Matrix_multiply({Matrix.Conjugate(e_out_sigma), OppR_T2g_Matrix, Matrix.Transpose(e_in_pi)})*sigma[2]/(S*(2*S - 1))
    
    OppRList = {OppR_A1g, OppR_T1u, OppR_Eg, OppR_T2g}
    
    -- For the first output file, we want to calculate all transition matrix elements of the
    -- effective operators to see their dependence on the scattering angle and the spin orientation.
    -- Of course, for cross polarized light, only off-diagonal operators T1u and T2g are non-zero.
    for j, operator in pairs(OppRList) do
    
        intensity_0 = psiList[1]*operator*psiList[1]
        intensity_1 = psiList[2]*operator*psiList[1]
        intensity_2 = psiList[3]*operator*psiList[1]
        
        intensity_0 = Complex.Re(Conjugate(intensity_0)*intensity_0)
        intensity_1 = Complex.Re(Conjugate(intensity_1)*intensity_1)
        intensity_2 = Complex.Re(Conjugate(intensity_2)*intensity_2)
        
        OUT_k:write(string.format("%.15f %.15f %.15f ", intensity_0, intensity_1, intensity_2))
    end
    
    -- To calculate the RIXS spectral function, we now need to relate the effective operators to the Boson operators
    -- in the linear spin wave theory. (Holstein-Primakoff transformation). As we have linear operators S+ but
    -- also quadratic operators SzS+ contributing to the single magnon excitation, we get different prefactors for their
    -- T1u and T2g and Eg contributions
    
    -- single magnon amplitude
    amplitude_1 =       math.sqrt(2*S)*psiList[2]*OppR_T1u*psiList[1]
                    +   math.sqrt(2*S)*S*psiList[2]*OppR_T2g*psiList[1]
                    +   math.sqrt(2*S)*S*psiList[2]*OppR_Eg*psiList[1]
    -- double magnonn amplitude
    amplitude_2 =       2*S*psiList[3]*OppR_T2g*psiList[1]
                    +   2*S*psiList[3]*OppR_Eg*psiList[1]
    
    -- As the Heisenberg Hamiltonian is diagonal in the Boson operators, we can calculate the intensies straight forward
    intensity_1 = Complex.Re(Conjugate(amplitude_1)*amplitude_1)
    intensity_2 = Complex.Re(Conjugate(amplitude_2)*amplitude_2)
    
    for e = 0, NE do
        
        local w = Emin + (Emax - Emin)/NE*e
                    
        OUT_w_k:write(string.format("%.15e ", i))
        OUT_w_k:write(string.format("%.15e ", w))
        -- this funciton calculates the single magnon spectral function. As we only have a single resonance
        -- this in principle is only a Lorenzian at w_k
        OUT_w_k:write(string.format("%.15e ", HFM_S1_w_k(w, k, hoppings_EuO, lattices_EuO, gamma, w_k)*intensity_1))
        -- For the two magnon spectral function, instead, we will observe a broad band.
        -- Exciting two fundamental particles at a particular momentum k leaves many possibilities
        -- which single particle momenta will add up to the total momentum k. Both particles could have different energies.
        -- Integrating over all possible combinations (here discretized by the grid points Nq for kx, ky, kz)
        -- will lead to a broad band instead of a single resonance. 
        -- This will take a LOT OF TIME!
        OUT_w_k:write(string.format("%.15e ", HFM_S2_w_k(w, k, hoppings_EuO, lattices_EuO, gamma, Nq, Nq, Nq)*intensity_2))
        OUT_w_k:write("\n")
    end

    OUT_k:write("\n")
    OUT_w_k:write("\n")
end

OUT_k:close()
OUT_w_k:close()


TimeEnd("Calculation")
TimePrint()
